unit allfuncs;

// Copyright  1999 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking

// A growing collection of functions that are generally useful
// in this and other programs.

interface
uses Windows, SysUtils, Forms, Classes, Graphics, stdCtrls;

type
  gdExeType = (gdNO, gdMZ, gdNE, gdPE);
  gvStrType = (vsCompanyName, vsFileDescription, vsFileVersion,
    vsInternalName, vsLegalCopyright, vsOriginalFilename,
    vsProductName, vsProductVersion);
  function GetDescription(const fname : String;
    var Typ : gdExeType) : String;
  function GetDescriptionEZ(const fname : String) : String;
  function GetVersionString(const fname : String; vs : gvStrType) :
    String;
  function IsPreRelease(const fname : String) : Boolean;
  function FinalSlash(const S : String) : String;
  procedure GetPosFmINI(const S : String; F : TForm; Siz : Boolean);
  procedure SetPosToINI(const S : String; F : TForm; Siz : Boolean);
  function GetSpecialPath(H : HWnd; nFolder : Integer) : String;
  procedure WrapLabel(const S : String; MaxW, MaxH, Play : Integer;
    CV : TCanvas; L : TLabel);
  function ReadALink(const S : String; pPath, pArgs, pWork, pIcon,
    pDesc : PChar; i : pInteger) : Boolean;
  function MakeALink(const S : String; pPath, pArgs, pWork, pIcon,
    pDesc : PChar; i : pInteger) : Boolean;


implementation
USES shlobj, comobj, activex, IniFiles;

TYPE
  {details of DOS header structure aren't important to us,
   so we just read the first 64 (40h) bytes}
  TDOSHeader = ARRAY[0..$3F] OF Byte;

  TNEHeader = RECORD
    {New Executable file header}
    Signature                      : Word;
    LinkerVersion,
    LinkerRevision                 : Byte;
    EntryTableRelOffset,
    EntryTableLength               : Word;
    Reserved                       : LongInt;
    Flags,
    AutomaticDSegNumber,
    LocalHeapSize,
    StackSize                      : Word;
    CSIP,
    SSSP                           : Pointer;
    SegmentTableNumEntries,
    ModuleReferenceTableNumEntries,
    NonresidentNameTableSize,
    SegmentTableRelOffset,
    ResourceTableRelOffset,
    ResidentNameTableRelOffset,
    ModuleReferenceTableRelOffset,
    ImportedNameTableRelOffset     : Word;
    NonresidentNameTableOffset     : LongInt;
    NumberOfMovableEntryPoints,
    ShiftCount,
    NumberOfResourceSegments       : Word;
    TargetOS,
    AdditionalInfo                 : Byte;
    FastLoadAreaOffset,
    FastLoadAreaSectors,
    Reserved2,
    ExpectedWindowsVersion         : Word;
  END;

function GetVersionString(const fname : String; vs : gvStrType) :
  String;
VAR
  verInfoSize : Integer;
  DummyGets0  : DWORD;
  verInfoBuff : Pointer;
  P,
  langCodeP   : Pointer;
  langCode    : DWORD;
  valName     : String;
  holdLen     : UInt; // 4/15/99
  vsLen       : UInt;
  verStrP     : Pointer;
  verStr      : PChar;
begin
  Result := '';
  // Get size of version info
  verInfoSize := GetFileVersionInfoSize(PChar(fName), DummyGets0);
  // Allocate buffer
  GetMem(verInfoBuff, verInfoSize);
  try
    // Get version info into buffer
    IF NOT GetFileVersionInfo(PChar(fName), 0, verInfoSize,
      verInfoBuff) THEN Exit;
    valName := '';
    // To read version resource strings we must know the language
    // charset code. The version info itself stores an array of
    // these codes. We use the first in which the language portion
    // is 0409 ("English (United States)"). The charset portion of
    // the code is commonly 04E4 ("Windows 3.1 Latin 1 (US,
    // Western Europe") or 4B0 ("Unicode (BMP of ISO 10646)")
    // If none match 0409, we try for any English
    IF VerQueryValue(verInfoBuff, '\VarFileInfo\Translation',
      langCodeP, vsLen) THEN
      begin
        P := langCodeP;
        holdLen := vsLen; // 4/15/99
        // Try to find 0409 ("English (United States)")
        WHILE vsLen > 0 DO
          begin
            langCode := PDWORD(P)^;
            IF LoWord(langCode) = $0409 THEN
              begin
                valName := Format('\StringFileInfo\%.04x%.04x\',
                  [LoWord(langCode), HiWord(langCode)]);
                Break;
              end;
            Dec(vsLen, 4);
            Inc(Integer(P), 4);
          end;
        vsLen := holdLen; // 4/15/99
        IF valName = '' THEN // try for any English
          begin
            P := langCodeP;
            WHILE vsLen > 0 DO
              begin
                langCode := PDWORD(P)^;
                IF LoByte(LoWord(langCode)) = $09 THEN
                  begin
                    valName := Format('\StringFileInfo\%.04x%.04x\',
                      [LoWord(langCode), HiWord(langCode)]);
                    Break;
                  end;
                Dec(vsLen, 4);
                Inc(Integer(P), 4);
              end;
          end;
        // If no English at all, just use the first
        IF valName = '' THEN
          begin
            langCode := PDWORD(langCodeP)^;
            valName := Format('\StringFileInfo\%.04x%.04x\',
              [LoWord(langCode), HiWord(langCode)]);
          end;
      end;
    IF valName = '' THEN valName := '\StringFileInfo\040904E4\';
    CASE vs OF
      vsCompanyName       : valName := valName + 'CompanyName';
      vsFileDescription   : valName := valName + 'FileDescription';
      vsFileVersion       : valName := valName + 'FileVersion';
      vsInternalName      : valName := valName + 'InternalName';
      vsLegalCopyright    : valName := valName + 'LegalCopyright';
      vsOriginalFilename  : valName := valName + 'OriginalFilename';
      vsProductName       : valName := valName + 'ProductName';
      vsProductVersion    : valName := valName + 'ProductVersion';
    END;
    // Now request the actual string
    IF VerQueryValue(verInfoBuff, PChar(valName), verStrP,
      vsLen) THEN
      begin
        verStr := StrNew(PChar(verStrP));
        try
          Result := StrPas(verStr);
        finally
          StrDispose(verStr);
        end;
      end;
  finally
    FreeMem(verInfoBuff);
  end;
end;

function IsPreRelease(const fname : String) : Boolean;
VAR
  verInfoSize : Integer;
  DummyGets0  : DWORD;
  verInfoBuff : Pointer;
  vsLen       : UInt;
  P           : Pointer;
begin
  Result := False;
  // Get size of version info
  verInfoSize := GetFileVersionInfoSize(PChar(fName), DummyGets0);
  // Allocate buffer
  GetMem(verInfoBuff, verInfoSize);
  try
    // Get version info into buffer
    IF NOT GetFileVersionInfo(PChar(fName), 0, verInfoSize,
      verInfoBuff) THEN Exit;
    // Now request the actual string
    IF VerQueryValue(verInfoBuff, '\', P, vsLen) THEN
      begin
        Result := PVSFIXEDFILEINFO(P)^.dwFileFlags AND
          VS_FF_PRERELEASE > 0;
      end;
  finally
    FreeMem(verInfoBuff);
  end;
end;

function GetDescription(const fname : String;
  var Typ : gdExeType) : String;
// Return description from header for NE (16-bit)
// Return versioninfo description for PE (32-bit)
// Return null string otherwise
VAR
  F : File;
  W, N          : Word;
  B             : Byte;
  DH            : TDOSHeader;
  NEOffset      : Word; {Offset of NE header}
  NE            : TNEHeader;
  Desc          : ARRAY[0..MAX_PATH] OF Char;

  PROCEDURE OpenForReadOnly(VAR F : File; FName : String);
  {The calling routine must handle any exception caused
   by the attempt to open the file}
  VAR OldFileMode : Byte;
  BEGIN
    AssignFile(F, FName);
    OldFileMode := FileMode;
    FileMode := 0; {read-only}
    Reset(F,1);
    FileMode := OldFileMode;
  END;

begin
  typ := gdNO;
  Result := '';
  try
    OpenForReadOnly(F, fname);
    try
      BlockRead(F, DH, SizeOf(DH));
      Move(DH[0], N, 2);
      Move(DH[$18], W, 2);
      IF N = $5A4D THEN typ := gdMZ;
      IF (N <> $5A4D{MZ}) OR ((W < $40) AND (W > 0)) THEN
        // No DOS executable header. Shouldn't happen!
        Exit;
      Move(DH[$3C], NEOffset, 2);
      Seek(F, NEOffset);
      BlockRead(F, NE, SizeOf(NE));
      IF NE.Signature = $454E{NE} THEN
        begin
          // 16-bit "New Executable"
          typ := gdNE;
          Seek(F, NE.NonresidentNameTableOffset);
          BlockRead(F, B, 1);
          FillChar(Desc, SizeOf(Desc),0);
          BlockRead(F, Desc, B);
          IF StrLen(Desc) <> 0 THEN
            Result := StrPas(Desc);
        end
      ELSE IF NE.Signature = $4550{PE} THEN
        begin
          // 32-bit "Portable Executable"
          typ := gdPE;
          Result := GetVersionString(fName, vsFileDescription);
        end;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    // Let it pass - result is blank
  end;
end;

function GetDescriptionEZ(const fname : String) : String;
VAR dummy : gdExeType;
begin
  Result := GetDescription(fname, dummy);
end;

function FinalSlash(const S : String) : String;
begin
  IF (S <> '') AND (S[Length(S)] <> '\') THEN
    Result := S + '\'
  ELSE Result := S;
end;

procedure GetPosFmINI(const S : String; F : TForm; Siz : Boolean);
VAR R : TRect;
begin
  SystemParametersInfo(SPI_GETWORKAREA, 0, @R, 0);
  WITH TIniFile.Create(S) DO
  try
    IF Siz THEN
      begin
        F.Width := ReadInteger(F.Name + ' Settings', 'Width',
          F.Width);
        F.Height := ReadInteger(F.Name + ' Settings', 'Height',
          F.Height);
        WITH F, Constraints DO
          begin
            IF Width < MinWidth THEN Width := MinWidth;
            IF (MaxWidth > 0) AND (Width > MaxWidth) THEN
              Width := MaxWidth;
            IF Height < MinHeight THEN Height := MinHeight;
            IF (MaxHeight > 0) AND (Height > MaxHeight) THEN
              Height := MaxHeight;
          end;
      end;
    F.Left := ReadInteger(F.Name + ' Settings', 'Left',
      (R.Right - R.Left - F.Width) DIV 2);
    F.Top := ReadInteger(F.Name + ' Settings', 'Top',
      (R.Bottom - R.Top - F.Height) DIV 2);
    IF F.Left < 0 THEN F.Left := 0;
    IF F.Top < 0 THEN F.Top := 0;
    IF F.Left + F.Width > R.Right THEN
      F.Left := R.Right - F.Width;
    IF F.Top + F.Height > R.Bottom THEN
      F.Top := R.Bottom - F.Height;
  finally
    Free;
  end;
end;

procedure SetPosToINI(const S : String; F : TForm; Siz : Boolean);
begin
  WITH TInIFile.Create(S) DO
  try
    IF Siz THEN
      begin
        WriteInteger(F.Name + ' Settings', 'Width', F.Width);
        WriteInteger(F.Name + ' Settings', 'Height', F.Height);
      end;
    WriteInteger(F.Name + ' Settings', 'Left', F.Left);
    WriteInteger(F.Name + ' Settings', 'Top', F.Top);
  finally
    Free;
  end;
end;

function GetSpecialPath(H : HWnd; nFolder : Integer) : String;
VAR
  thePIDL : PItemIDList;
  theBUFF : ARRAY[0..MAX_PATH] OF Char;
begin
  IF SHGetSpecialFolderLocation(H, nfolder, thePIDL) = NOERROR THEN
    begin
      IF SHGetPathFromIDList(thePIDL, theBUFF) THEN
        Result := StrPas(theBuff)
      ELSE Result := '';
    end
  ELSE Result := '';
end;

procedure WrapLabel(const S : String; MaxW, MaxH, Play : Integer;
  CV : TCanvas; L : TLabel);
// Sets the label's caption to the specified string, with
// sensible word-wrap even if the string contains no spaces.
// Adjust the label's height as necessary
VAR
  WorkS, TempS : String;
  Pos, N       : Integer;
  StartPos     : Integer;
  R            : TRect;
begin
  TempS := '';
  WorkS := S;
  // Make a guess as to where the string should break
  IF Length(WorkS) = 0 THEN StartPos := 0
  ELSE
    StartPos := (MaxW * Length(WorkS)) DIV CV.TextWidth(WorkS);
  // Build a new string, inserting spaces as needed to
  // force line breaks
  WHILE CV.TextWidth(WorkS) > MaxW DO
    begin
      Pos := StartPos;
      WHILE CV.TextWidth(Copy(WorkS, 1, Pos)) >= MaxW DO Dec(Pos);
      WHILE CV.TextWidth(Copy(WorkS, 1, Pos)) <  MaxW DO Inc(Pos);
      Dec(Pos);
      N := 0;
      WHILE (N < Pos-1) AND (N < Play) DO
        IF WorkS[Pos-N] IN [' ','.',',',';',':','/','\','-'] THEN
          Break
        ELSE Inc(N);
      IF WorkS[Pos-N] IN [' ','.',',',';',':','/','\','-'] THEN Pos := Pos-N;
      TempS := TempS + Copy(WorkS, 1, Pos);
      IF WorkS[Pos] <> ' ' THEN TempS := TempS + ' ';
      Delete(WorkS, 1, Pos);
    end;
  TempS := TempS + WorkS;
  // Use DrawText to calculate the necessary size
  FillChar(R, SizeOf(R), 0);
  R.Right := MaxW;
  DrawText(CV.Handle, PChar(TempS), Length(TempS), R,
    DT_CALCRECT OR DT_WORDBREAK OR DT_NOPREFIX);
  L.WordWrap := True;
  L.Width := R.Right;  
  IF R.Bottom < MaxH THEN L.Height := R.Bottom
  ELSE L.Height := MaxH;
  L.Caption := TempS;
end;

function ReadALink(const S : String; pPath, pArgs, pWork, pIcon,
  pDesc : PChar; i : pInteger) : Boolean;
// Read the shortcut named in the S argument, and fill any
// non-nil arguments with data from the shortcut. Each non-nil
// PChar must be big enough to hold MAX_PATH characters
VAR             
  vUNK   : IUnknown;
  vISL   : IShellLink;
  vIPF   : IPersistFile;
  fNameW : ARRAY[0..MAX_PATH] OF WideChar;
  TW     : TWin32FindData;
begin
  Result := False;
  try
    StringToWideChar(S, fNameW, MAX_PATH);
    vUNK := CreateComObject(CLSID_ShellLink);
    vISL := vUNK AS IShellLink;
    vIPF := vUNK AS IPersistFile;
    IF vIPF.Load(@fNameW, STGM_READ) <> S_OK THEN Exit;
    Result := True;
    IF pPath <> nil THEN
      IF vISL.GetPath(pPath, MAX_PATH, TW, 0) <> S_OK THEN
        pPath[0] := #0;
    IF pArgs <> nil THEN
      IF vISL.GetArguments(pArgs, MAX_PATH) <> S_OK THEN
        pArgs[0] := #0;
    IF pWork <> nil THEN
      IF vISL.GetWorkingDirectory(pWork, MAX_PATH) <> S_OK THEN
        pWork[0] := #0;
    IF (pIcon <> nil) AND (i <> nil) THEN
      IF vISL.GetIconLocation(pIcon, MAX_PATH, i^) <> S_OK THEN
        pIcon[0] := #0;
    IF pDesc <> nil THEN
      IF vISL.GetDescription(PDesc, MAX_PATH) <> S_OK THEN
        pDesc[0] := #0;
  except
    ON Exception DO;
  end;
end;

function MakeALink(const S : String; pPath, pArgs, pWork, pIcon,
  pDesc : PChar; i : pInteger) : Boolean;
// Create the shortcut named in the S argument, and load its
// data from the non-nil arguments. Return True if successful.
VAR
  vUNK : IUnknown;
  vISL : IShellLink;
  vIPF : IPersistFile;
  fNameW : ARRAY[0..MAX_PATH] OF WideChar;
begin
  Result := False;
  try
    StringToWideChar(S, fNameW, MAX_PATH);
    vUNK := CreateComObject(CLSID_ShellLink);
    vISL := vUNK AS IShellLink;
    vIPF := vUNK AS IPersistFile;
    IF pPath <> nil THEN
      IF vISL.SetPath(pPath) <> S_OK THEN Exit;
    IF pArgs <> nil THEN
      IF vISL.SetArguments(pArgs) <> S_OK THEN Exit;
    IF (pIcon <> nil) AND (i <> nil) THEN
      IF vISL.SetIconLocation(pIcon, i^) <> S_OK THEN Exit;
    IF pWork <> nil THEN
      IF vISL.SetWorkingDirectory(pWork) <> S_OK THEN Exit;
    IF pDesc <> nil THEN
      IF vISL.SetDescription(pDesc) <> S_OK THEN Exit;
    IF  vIPF.Save(@fNameW, False) <> S_OK THEN Exit;
    Result := True;
  except
    ON Exception DO;
  end;
end;


end.
