unit mainu;

// Copyright  1999 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking

(*
ACCELERATOR (underlined) keys in use:
  Always     :  ABGHPRS
  first tab  :  ABDEGHPRSTV
  second tab :  ABDEGHNPRS
  third tab  :  ABCDEFGHLOPRST
*)

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, Menus, ActnList, ImgList, StdCtrls, ComCtrls,
  Buttons;

CONST
  MainMapName : PChar = 'PC Magazine Startup Cop Shared';
  MainMsgName : PChar = 'PC Magazine Startup Cop Message';
TYPE
  TCmdMode = (tcmNone, tcmClose, tcmLogoff, tcmRestart, tcmError);
  TShared = Record
    AppHandle : THandle;
    WinHandle : THandle;
    CmdLine   : ShortString;
    CmdMode   : TCmdMode;
  end;
VAR
  PSharedData  : ^TShared;
  CmdLineMsg   : Cardinal;
type
  TMainForm = class(TForm)
    ActionList1        : TActionList;
      acPack           : TAction;
      acDisable        : TAction;
      acEnable         : TAction;
      acExplain        : TAction;
      acApply          : TAction;
      acReset          : TAction;
    PopupMenu1         : TPopupMenu;
      mnuDisable       : TMenuItem;
      mnuEnable        : TMenuItem;
      mnuPack          : TMenuItem;
      N1               : TMenuItem;
      mnuExplain       : TMenuItem;
    Timer1             : TTimer;
    ImageList1         : TImageList;
    PageControl1       : TPageControl;
      TabSheet1        : TTabSheet;
        lvProgs        : TListView;
        pnlStartup     : TPanel;
          imgLite      : TImage;
          lblRed       : TLabel;
          lblYel       : TLabel;
          lblGrn       : TLabel;
          rbRemove     : TRadioButton;
          rbDisable    : TRadioButton;
          rbEnable     : TRadioButton;
          pnlHid       : TPanel;
            btnDelete  : TBitBtn;
          pnlLid       : TPanel;
            Image1     : TImage;
          btnExplain   : TButton;
      TabSheet2        : TTabSheet;
        ebSave         : TEdit;
        btnSaveEna     : TBitBtn;
        btnSaveDis     : TBitBtn;
      TabSheet3        : TTabSheet;
        pnlRestore     : TPanel;
          btnRestore   : TButton;
          btnLogoff    : TButton;
          btnRestart   : TButton;
          btnDelProf   : TButton;
          btnShortcut  : TButton;
          btnShortLog  : TButton;
          btnShortRest : TButton;
        pnlProfs       : TPanel;
          lbProfs      : TListBox;
        spltRest       : TSplitter;
        pnlItems       : TPanel;
          lblItems     : TLabel;
          lbItems      : TListBox;
    pnlMain            : TPanel;
      btnOK            : TButton;
      btnCancel        : TButton;
      btnApply         : TButton;
      btnReset         : TButton;
      btnHelp          : TButton;
      btnAbout         : TButton;
    procedure FormCreate         (Sender: TObject);
    procedure FormDestroy        (Sender: TObject);
    procedure FormActivate       (Sender: TObject);
    procedure acPackExecute      (Sender: TObject);
    procedure acDisableExecute   (Sender: TObject);
    procedure acEnableExecute    (Sender: TObject);
    procedure acExplainExecute   (Sender: TObject);
    procedure acApplyExecute     (Sender: TObject);
    procedure acApplyUpdate      (Sender: TObject);
    procedure acResetExecute     (Sender: TObject);
    procedure Timer1Timer        (Sender: TObject);
    procedure PageControl1Change (Sender: TObject);
    procedure PanelMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure lvProgsChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure lvProgsMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure btnDeleteExecute   (Sender: TObject);
    procedure ebSaveChange       (Sender: TObject);
    procedure ebSaveKeyPress     (Sender: TObject; var Key: Char);
    procedure btnSaveClick       (Sender: TObject);
    procedure btnRestoreClick    (Sender: TObject);
    procedure btnShortcutClick   (Sender: TObject);
    procedure btnDelProfClick    (Sender: TObject);
    procedure lbProfsClick       (Sender: TObject);
    procedure lbProfsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure lbItemsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure btnOKClick         (Sender: TObject);
    procedure btnCancelClick     (Sender: TObject);
    procedure btnHelpClick       (Sender: TObject);
    procedure btnAboutClick      (Sender: TObject);
  private
    { Private declarations }
    lG,                // stoplight image w/ green
    lY,                // stoplight image w/ yellow
    lR,                // stoplight image w/ red
    lX      : TBitmap; // stoplight image w/ none
    IniName : String;  // name for INI file
    NumEna,            // number of green-light items
    NumDis  : Integer; // number of yellow-light items
    Opening : Boolean; // used in "hidden button" animation
    procedure ProcessCmdLine;
    procedure LoadTheList;
    procedure MakeChanges(TellTrouble : Boolean);
    function IsSysTray(TLI : TListItem) : Boolean;
    function RestoreProfile(const Sec : String) : Integer;
  public
    { Public declarations }
    procedure DefaultHandler(VAR Message); override;
  end;

var
  MainForm: TMainForm;

implementation
USES AllFuncs, expFormu, AboutBox, scopshar,
  IniFiles, Registry, CommCtrl, activex, comobj, shlobj,
  fileCtrl, MMSystem;

{$R *.DFM}
{$R MORERES}

procedure TMainForm.FormCreate(Sender: TObject);
VAR
  TS : TStringList;
  N  : Integer;
  S  : String;
begin
  // Create the stoplight bitmaps and load their images
  lG        := TBitmap.Create;
  lG.Handle := LoadBitmap(hInstance, MakeIntResource(1));
  lY        := TBitmap.Create;
  lY.Handle := LoadBitmap(hInstance, MakeIntResource(2));
  lR        := TBitmap.Create;
  lR.Handle := LoadBitmap(hInstance, MakeIntResource(3));
  lX        := TBitmap.Create;
  lX.Handle := LoadBitmap(hInstance, MakeIntResource(4));
  // Initialize variables
  imgLite.Picture.Bitmap  := lX;
  PageControl1.ActivePage := TabSheet1;
  IniName := ChangeFileExt(Application.ExeName, '.INI');
  Width := 600; Height := 400;
  GetPosFmINI(IniName, Self, True);
  Application.HelpFile := ChangeFileExt(Application.ExeName, '.HLP');
  Icon := Application.Icon;
  // Load the list of startup programs
  LoadTheList;
  // Load profile names from .INI file
  TS := TStringList.Create;
  try
    WITH TIniFile.Create(IniName) DO
    try
      pnlProfs.Width := ReadInteger(Self.name + ' Settings',
        'Panel4Width', 96);
      ReadSections(TS);
      lbProfs.Items.Clear;
      FOR N := 0 TO TS.Count-1 DO
        IF Pos('profile', TS[N]) = 1 THEN
          begin
            S := Copy(TS[N], 8, 255);
            IF ReadInteger(TS[N], 'State', 1) = 1 THEN
              lbProfs.Items.AddObject(S, Pointer(1))
            ELSE
              lbProfs.Items.AddObject(S, Pointer(2));
          end;
    finally
      Free;
    end;
  finally
    TS.Free;
  end;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  SetPosToINI(IniName, Self, True);
  WITH TIniFile.Create(iniName) DO
  try
    WriteInteger(Self.name + ' Settings', 'Panel4Width',
      pnlProfs.Width);
  finally
    Free;
  end;
end;

procedure TMainForm.FormActivate(Sender: TObject);
begin
  ProcessCmdLine;
end;

procedure TMainForm.acPackExecute(Sender: TObject);
begin
  IF lvProgs.Selected = nil THEN Exit;
  IF IsSysTray(lvProgs.Selected) THEN Exit;
  // If this item was "ready for removal" and has not had
  // any change applied, restore its "ready for removal"
  // state. Otherwise, just mark for removal
  IF lvProgs.Selected.Data = Pointer(4) THEN
    lvProgs.Selected.StateIndex := 4
  ELSE lvProgs.Selected.StateIndex := 3;
  imgLite.Picture.Bitmap := lR;
  lvProgs.SetFocus;
end;

procedure TMainForm.acDisableExecute(Sender: TObject);
begin
  IF lvProgs.Selected = nil THEN Exit;
  IF IsSysTray(lvProgs.Selected) THEN Exit;
  lvProgs.Selected.StateIndex := 2;
  imgLite.Picture.Bitmap := lY;
  lvProgs.SetFocus;
end;

procedure TMainForm.acEnableExecute(Sender: TObject);
begin
  IF lvProgs.Selected = nil THEN Exit;
  IF IsSysTray(lvProgs.Selected) THEN Exit;
  lvProgs.Selected.StateIndex := 1;
  imgLite.Picture.Bitmap := lG;
  lvProgs.SetFocus;
end;

procedure TMainForm.acExplainExecute(Sender: TObject);
begin
  IF lvProgs.Selected = NIL THEN Exit;
  WITH ExpForm DO
    begin
      SetItem(lvProgs.Selected);
      Show;
    end;
end;

procedure TMainForm.acApplyExecute(Sender: TObject);
begin
  MakeChanges(True);
  LoadTheList;
end;

procedure TMainForm.acApplyUpdate(Sender: TObject);
// Enable Apply only if at least one item has been changed
VAR
  N    : Integer;
  Same : Boolean;
begin
  Same := True;
  FOR N := 0 TO lvProgs.Items.Count-1 DO
    WITH lvProgs.Items[N] DO
      IF Pointer(StateIndex) <> Data THEN
        begin
          Same := False;
          Break;
        end;
  acApply.Enabled := NOT Same;
end;

procedure TMainForm.acResetExecute(Sender: TObject);
begin
  LoadTheList;
  acApply.Enabled := False;
end;

procedure TMainForm.Timer1Timer(Sender: TObject);
// If Opening is true, each timer tick moves the lid-opening
// animation forward; otherwise each timer tick moves it back.
// When animation in either direction is complete, the timer
// is disabled. It's possible to change direction in the middle
// of the animation.

  procedure SoundClick;
  begin
    PlaySound(MakeIntResource(5), hInstance, SND_RESOURCE OR
      SND_ASYNC);
  end;

  procedure SoundWhirr;
  begin
    PlaySound(MakeIntResource(6), hInstance, SND_RESOURCE OR
      SND_ASYNC OR SND_LOOP);
  end;

begin
  IF Opening THEN
    begin
      IF pnlLid.BevelInner = bvNone THEN
        pnlLid.BevelInner := bvRaised     // lid appears
      ELSE IF pnlLid.BevelOuter = bvNone THEN
        begin
          pnlLid.BevelOuter := bvRaised;  // lid grows
          SoundWhirr;                     // whirr sound starts
        end
      ELSE IF pnlLid.Top = pnlHid.Top THEN
        pnlLid.Top := pnlHid.Top-1        // lid moves up 1 pixel
      ELSE IF pnlLid.Left > pnlHid.left + pnlHid.Width THEN
        Timer1.Enabled := False           // stop animation
      ELSE IF pnlLid.Left+4 > pnlHid.left + pnlHid.Width THEN
        begin
          SoundClick;                     // click!
          pnlLid.Left := pnlLid.Left+4;   // move into final position
        end
      ELSE pnlLid.Left := pnlLid.Left+4;  // move to right
    end
  ELSE
    begin
      IF pnlLid.Left > pnlHid.left + pnlHid.Width THEN
        pnlLid.Left := pnlLid.Left-4      // move to left
      ELSE IF pnlLid.Left+4 > pnlHid.left + pnlHid.Width THEN
        begin
          pnlLid.Left := pnlLid.Left-4;   // move to left
          SoundWhirr;                     // whirr sound starts
        end
      ELSE IF pnlLid.Left > pnlHid.Left THEN
        pnlLid.Left := pnlLid.Left-4      // move to left
      ELSE IF pnlLid.Top <> pnlHid.Top THEN
        pnlLid.Top := pnlHid.Top          // lid moves down 1 pixel
      ELSE IF pnlLid.BevelOuter = bvRaised THEN
        pnlLid.BevelOuter := bvNone       // lid shrinks
      ELSE IF pnlLid.BevelInner = bvRaised THEN
        begin
          pnlLid.BevelInner := bvNone;    // lid disappears
          SoundClick;                     // click!
        end
      ELSE
        begin
          Timer1.Enabled := False;        // stop animation
        end;
    end;
end;

procedure TMainForm.PageControl1Change(Sender: TObject);
VAR N : Integer;
begin
  IF PageControl1.ActivePage = TabSheet1 THEN
    ActiveControl := lvProgs
  ELSE IF PageControl1.ActivePage = TabSheet2 THEN
    begin
      // NumDis is the number of programs that are disabled or
      // marked to be disabled. NumEna is the number that are
      // enabled or marked to be enabled.
      NumDis := 0;
      NumEna := 0;
      FOR N := 0 TO lvProgs.Items.Count-1 DO
        begin
          IF IsSysTray(lvProgs.Items[N]) THEN Continue;
          CASE lvProgs.Items[N].StateIndex OF
            1 : Inc(NumEna);
            2 : Inc(NumDis);
          END;
        end;
      btnSaveEna.Enabled := (NumEna > 0) AND (ebSave.Text <> '');
      btnSaveDis.Enabled := (NumDis > 0) AND (ebSave.Text <> '');
    end;
end;

procedure TMainForm.PanelMouseMove(Sender: TObject; Shift:
  TShiftState; X, Y: Integer);
// This enables flyover hints for disabled buttons
VAR
  P : TPoint;
  N : Integer;
begin
  P := Point(X,Y);
  WITH Sender AS TWinControl DO
    begin
      Hint := '';
      FOR N := 0 TO ControlCount-1 DO
        IF Controls[N] IS TWinControl THEN
          IF PtInRect(TWinControl(Controls[N]).BoundsRect, P) THEN
            Hint := TWinControl(Controls[N]).Hint;
      IF Hint = '' THEN
        Application.CancelHint;
    end;
end;

procedure TMainForm.lvProgsChange(Sender: TObject; Item: TListItem;
  Change: TItemChange);
begin
  // If nothing is selected...
  IF lvProgs.Selected = nil THEN
    begin
      // Disable selection-action items
      acPack.Enabled         := False;
      acDisable.Enabled      := False;
      acEnable.Enabled       := False;
      btnDelete.Enabled      := False;
      acExplain.Enabled      := False;
      // Clear checkmarks
      acPack.Checked         := False;
      acDisable.Checked      := False;
      acEnable.Checked       := False;
      // Show stoplight with NO lights
      imgLite.Picture.Bitmap := lX;
      // Close the hidden button
      Opening                := False;
      Timer1.Enabled         := True;
    end
  // If the system tray applet is selected...
  ELSE IF IsSysTray(lvProgs.Selected) THEN
    begin
      // Disable state-change buttons
      acPack.Enabled     := False;
      acDisable.Enabled  := False;
      acEnable.Enabled   := False;
      btnDelete.Enabled  := False;
      // Enable explanation and show check
      acExplain.Enabled  := True;
      acEnable.Checked   := True;
      // Close the hidden button
      Opening            := False;
      Timer1.Enabled     := True;
    end
  // If some other startup item is selected...
  ELSE
    WITH lvProgs.Selected DO
      begin
        // Enable the selection-action items
        acPack.Enabled    := True;
        acDisable.Enabled := True;
        acEnable.Enabled  := True;
        acExplain.Enabled := True;
        // If "ready for removal", enable Delete button
        btnDelete.Enabled := StateIndex = 4;
        // Close / open the hidden button
        Opening           := btnDelete.Enabled;
        Timer1.Enabled    := True;
        // Check off the appropriate radiobutton
        acEnable.Checked  := StateIndex = 1;
        acDisable.Checked := StateIndex = 2;
        acPack.Checked    := StateIndex >= 3;
        // Show the corresponding stoplight image
        CASE StateIndex OF
          1 :  imgLite.Picture.Bitmap := lG;
          2 :  imgLite.Picture.Bitmap := lY;
          ELSE imgLite.Picture.Bitmap := lR;
        END;
      end;
end;

procedure TMainForm.lvProgsMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
// Without this method, right-clicking on the *icon* for
// an item wouldn't move the selection and focus, which
// could cause confusion
begin
  WITH Sender AS TListView DO
    begin
      Selected    := GetItemAt(X,Y);
      ItemFocused := Selected;
    end;
end;

procedure TMainForm.btnDeleteExecute(Sender: TObject);
begin
  IF lvProgs.Selected = nil THEN Exit;
  // Only mark for deletion if the item is already
  // marked as "ready for removal"
  IF lvProgs.Selected.StateIndex <> 4 THEN Exit;
  IF lvProgs.Selected.Data <> Pointer(4) THEN Exit;
  lvProgs.Selected.StateIndex := 5;
  lvProgs.SetFocus;
end;

procedure TMainForm.ebSaveChange(Sender: TObject);
// For simplicity, profile names must consist only of letters,
// numbers, and spaces. This method handles stripping out
// wrong characters when a value is pasted in.
VAR
  P, N : Integer;
  S    : String;
  chg  : Boolean;
begin
  P   := ebSave.selStart;
  S   := ebSave.Text;
  chg := False;
  FOR N := Length(S) DOWNTO 1 DO
    begin
      CASE S[N] OF
        '0'..'9', 'a'..'z', 'A'..'Z', ' ' : ;
        ELSE
          begin
            Delete(S, N, 1);
            IF N < P THEN Dec(P);
            chg := True;
          end;
      END;
    end;
  IF chg THEN
    begin
      ebSave.Text      := S;
      ebSave.SelStart  := P;
      ebSave.SelLength := 0;
    end;
  btnSaveEna.Enabled := (NumEna > 0) AND (ebSave.Text <> '');
  btnSaveDis.Enabled := (NumDis > 0) AND (ebSave.Text <> '');
end;

procedure TMainForm.ebSaveKeyPress(Sender: TObject; var Key: Char);
// For simplicity, profile names must consist only of letters,
// numbers, and spaces. This method suppresses wrong characters.
begin
  CASE Key OF
    '0'..'9', 'a'..'z', 'A'..'Z', ' ', #8 : ;
    ELSE Key := #0;
  END;
end;

procedure TMainForm.btnSaveClick(Sender: TObject);
// This method is used by both the btnSaveEna and btnSaveDis
// buttons. The former has its Tag set to 1, the latter to 2.
const
  enname : ARRAY[1..2] OF String[8] = ('enabled', 'disabled');
VAR
  Sec    : String;
  N, Idx : Integer;
  EnDis  : Integer;
begin
  EnDis := (Sender AS TBitBtn).Tag;
  Sec   := 'profile'+ebSave.Text;
  WITH TIniFile.Create(IniName) DO
  try
    Idx := lbProfs.Items.IndexOf(ebSave.Text);
    IF Idx >= 0 THEN
      begin
        MessageBeep(MB_ICONQUESTION);
        IF MessageBox(Handle, PChar(Format('The profile %s already '+
          'exists. Do you want to overwrite it?', [ebSave.Text])),
          'Startup Cop', MB_YESNO OR MB_ICONQUESTION) <> idYES THEN
            Exit;
        EraseSection(Sec);
        lbProfs.Items.Objects[Idx] := Pointer(EnDis);
      end
    ELSE
      Idx := lbProfs.Items.AddObject(ebSave.Text, Pointer(EnDis));
    WriteInteger(Sec, 'State', EnDis);
    FOR N := 0 TO lvProgs.Items.Count-1 DO
      WITH lvProgs.Items[N] DO
        IF StateIndex = EnDis THEN
          WriteString(Sec, Format('%d%s',[ImageIndex, Caption]),
            SubItems[3]);
    // Now select the just-saved profile
    lbProfs.ItemIndex := Idx;
    lbProfsClick(lbProfs);
  finally
    Free;
  end;
end;

procedure TMainForm.btnRestoreClick(Sender: TObject);
// Used by all three Restore buttons. Plain restore has Tag set to 0,
// logoff has Tag set to 1, restart has Tag set to 2.
VAR errS : String;
begin
  WITH lbProfs DO
    begin
      CASE RestoreProfile('profile'+Items[ItemIndex]) OF
        -1 : ErrS := 'does not exist';
        -2 : ErrS := 'is corrupted';
        -3 : ErrS := 'is empty';
        ELSE ErrS := '';
      END;
      IF ErrS <> '' THEN
        begin
          MessageBeep(MB_ICONHAND);
          MessageBox(Handle, PChar(Format('The profile "%s" %s.',
            [Items[ItemIndex], ErrS])), 'Startup Cop', MB_OK OR
            MB_ICONHAND);
          Exit;
        end;
    end;
  WITH Sender AS TControl DO
    IF Tag = 0 THEN
      begin
        PageControl1.ActivePage := TabSheet1;
      end
    ELSE
      begin
        MakeChanges(Tag=0);
        CASE Tag OF
          1 : ExitWindowsEx(EWX_LOGOFF, 0);
          2 : ExitWindowsEx(EWX_REBOOT, 0);
        END;
        Close;
      end;
end;

procedure TMainForm.btnShortcutClick(Sender: TObject);
// Create a shortcut on the desktop that will launch Startup Cop
// using the specified profile. Used by all three Shortcut
// buttons. Plain shortcut has tag set to 0, w/ logoff has
// Tag set to 1, w/ restart has Tag set to 2
const
  pref : ARRAY[0..2] OF String[4] = ('', '[L] ', '[R] ');
//1.01  arg2 : ARRAY[0..2] OF String[3] = ('', ' /L', '/ R');
  arg2 : ARRAY[0..2] OF String[3] = ('', ' /L', '/ R'); //1.01
  what : ARRAY[0..2] OF String = ('.', #13#10#13#10'It will then lo'+
    'g you out of Windows', #13#10#13#10'It will then restart Windows.');
VAR
  Args    : String;
  Lnkname : String;
  I       : Integer;
begin
  I := (Sender AS TButton).Tag;
  IF NOT (I IN [0..2]) THEN I := 0;
  // Get the folder that holds desktop shortcuts
  LnkName := GetSpecialPath(Handle, CSIDL_DESKTOPDIRECTORY);
  WITH lbProfs DO
    begin
      LnkName := Format('%s%s%s.lnk', [FinalSlash(LnkName),
        pref[I], Items[ItemIndex]]);
      Args := Format('"%s"%s', [Items[ItemIndex], arg2[I]]);
      IF MakeALink(LnkName, PChar(Application.ExeName), PChar(Args),
        PChar(ExtractFileDir(Application.ExeName)), nil, 'Shortcut '+
        'created by PC Magazine''s Startup Cop utility', nil) THEN
          begin
            MessageBox(Handle, PChar(Format('Startup Cop has create'+
            'd a Desktop shortcut named "%s%s".'#13#10#13#10'When y'+
            'ou launch this shortcut, Startup Cop will restore and '+
            'apply the profile named "%s"%s',
            [pref[I], Items[ItemIndex], Items[ItemIndex], what[I]])),
            'Startup Cop', MB_OK OR MB_ICONINFORMATION);
          end
        ELSE
          begin
            MessageBeep(MB_ICONHAND);
            MessageBox(Handle, PChar(Format('Error. Startup Cop was'+
              ' not able to create a Desktop shortcut for the profi'+
              'le "%s".',
              [Items[ItemIndex]])),
            'Startup Cop', MB_OK OR MB_ICONHAND);
          end;
    end;
end;

procedure TMainForm.btnDelProfClick(Sender: TObject);
VAR Idx : Integer;
begin
  IF MessageBox(Handle, PChar(Format('Delete profile "%s"?',
    [lbProfs.Items[lbProfs.ItemIndex]])), 'Startup Cop',
    MB_YESNO OR MB_ICONQUESTION) <> idYes THEN Exit;
  WITH TIniFile.Create(IniName) DO
  try
    // Erase the INI file section representing the profile
    EraseSection('profile'+lbProfs.items[lbProfs.ItemIndex]);
    Idx := lbProfs.ItemIndex;
    // Erase the profile from Startup Cop's displayed list
    lbProfs.Items.Delete(Idx);
    IF Idx >= lbProfs.Items.Count THEN Dec(Idx);
    lbProfs.ItemIndex := Idx;
    lbProfsClick(lbProfs);
  finally
    Free;
  end;
end;

procedure TMainForm.lbProfsClick(Sender: TObject);
VAR N : Integer;
begin
  WITH Sender AS TListBox DO
    IF ItemIndex = -1 THEN
      begin
        ebSave.Text := '';
        btnRestore.Enabled := False;
        lblItems.Caption := '';
        lbItems.Items.Clear;
      end
    ELSE
      begin
        ebSave.Text := Items[ItemIndex];
        btnRestore.Enabled := True;
        WITH TIniFile.Create(Ininame) DO
        try
          IF ReadInteger('profile'+Items[ItemIndex],
            'State', 1) = 1 THEN
            lblItems.Caption := 'Items to be enabled'
          ELSE
            lblItems.Caption := 'Items to be disabled';
          ReadSection('profile'+Items[ItemIndex], lbItems.Items);
          // The items are stored with a number from 0 to 6
          // prefixed to them, representing the location from
          // which they're launched. Strip off this number
          // before displaying them.
          FOR N := lbItems.Items.Count-1 DOWNTO 0 DO
            IF lbItems.Items[N][1] IN ['0'..'6'] THEN
              lbItems.Items[N] := Copy(lbItems.Items[N], 2, 255)
            ELSE lbItems.Items.Delete(N);
        finally
          Free;
        end;
      end;
  btnLogOff.Enabled    := btnRestore.Enabled;
  btnRestart.Enabled   := btnRestore.Enabled;
  btnDelProf.Enabled   := btnRestore.Enabled;
  btnShortcut.Enabled  := btnRestore.Enabled;
  btnShortLog.Enabled  := btnRestore.Enabled;
  btnShortRest.Enabled := btnRestore.Enabled;
end;

procedure TMainForm.lbProfsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
VAR R : TRect;
begin
  WITH Control AS TListBox, Canvas DO
    begin
      FillRect(Rect);
      // Draw a green or yellow light
      IF Integer(Items.Objects[Index]) = 1 THEN
        ImageList1.Draw(Canvas, Rect.Left, Rect.Top, 1)
      ELSE
        ImageList1.Draw(Canvas, Rect.Left, Rect.Top, 2);
      R      := Rect;
      R.Left := R.Left + 18;
      TextRect(R, R.Left, R.Top+2, Items[Index]);
    end;
end;

procedure TMainForm.lbItemsDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
// There's no point to selecting an item in this list, so
// always draw the items as-if not selected
begin
  WITH Control AS TListBox, Canvas DO
    begin
      Font.Color  := clWindowText;
      Brush.Color := clWindow;
      TextRect(Rect, Rect.Left, Rect.Top, Items[Index]);
    end;
end;

procedure TMainForm.btnOKClick(Sender: TObject);
begin
  IF acApply.Enabled THEN MakeChanges(True);
  Close;
end;

procedure TMainForm.btnCancelClick(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.btnHelpClick(Sender: TObject);
begin
  Application.HelpCommand(HELP_FINDER, 0);
end;

procedure TMainForm.btnAboutClick(Sender: TObject);
begin
  WITH TAboutForm.Create(Self) DO
  try
    ShowModal;
  finally
    Free;
  end;
end;

procedure TMainForm.ProcessCmdLine;
VAR
  Cmd, ErrS : String;
  Mode      : TCmdMode;
begin
  Cmd                  := PSharedData^.CmdLine;
  Mode                 := PSharedData^.CmdMode;
  PSharedData^.CmdLine := '';
  PSharedData^.CmdMode := tcmNone;
  IF Cmd <> '' THEN
    begin
      IF Mode = tcmError THEN
        begin
          MessageBeep(MB_ICONHAND);
          MessageBox(Handle,
            'Syntax: STARTCOP "profile name" [/L | /R]'#13#10#13#10+
            '  "profile name"'#9'An existing Startup Cop profile'+
            #13#10+
            '  /L'#9#9'Restore and apply the profile, then log off'+
            #13#10+
            '  /R'#9#9'Restore and apply the profile, then restart '+
            'Windows'#13#10#13#10+
            'If neither /L or /R is present, Startup Cop will termi'+
            'nate after'#13#10+
            'restoring and applying the profile',
            'Startup Cop Command-Line Syntax Error',
            MB_OK OR MB_ICONHAND);
          Exit;
        end;
      CASE RestoreProfile('profile'+Cmd) OF
        -1 : ErrS := 'does not exist';
        -2 : ErrS := 'is corrupted';
        -3 : ErrS := 'is empty';
        ELSE ErrS := '';
      END;
      IF ErrS <> '' THEN
        begin
          MessageBeep(MB_ICONHAND);
          MessageBox(Handle, PChar(Format('The profile "%s" %s.',
          [Cmd, ErrS])), 'Startup Cop', MB_OK OR
          MB_ICONHAND);
          Exit;
        end;
      MakeChanges(NOT (Mode IN [tcmLogOff, tcmRestart]));
      CASE Mode OF
        tcmLogOff  : begin
          ExitWindowsEx(EWX_LOGOFF, 0);
          PostMessage(Handle, CmdLineMsg, 1, 0);
        end;
        tcmRestart : ExitWindowsEx(EWX_REBOOT, 0);
        tcmClose   : PostMessage(Handle, CmdLineMsg, 1, 0);
        tcmNone    : begin
          PageControl1.ActivePage := TabSheet1;
          Show;
        end;
      END;
    end;
end;


procedure TMainForm.LoadTheList;
VAR
  Idx     : Integer;
  Already : TStringList;

  procedure AddAnItem(const cmd, vDesc : String; src : TStSrcType;
    Light :Integer; vExt : String);
  VAR LI : TListItem;
  begin
    IF cmd = '' THEN Exit;
    CASE Light OF
      1 : Inc(NumEna);
      2 : Inc(NumDis);
    END;
    LI := lvProgs.Items.Add;
    WITH LI DO
      begin
        Caption    := vDesc;
        ImageIndex := Integer(src);
        StateIndex := Light;
        IF Already.IndexOf(vDesc) > -1 THEN
          StateIndex := StateIndex + 10
        ELSE
          Already.Add(vDesc);
        Data       := Pointer(StateIndex);
        case Src OF
          sRMacS : begin
            SubItems.Add('1');
            SubItems.Add('All users');
            SubItems.Add('Windows startup');
          end;
          sILoad : begin
            SubItems.Add('2');
            SubItems.Add('All users');
            SubItems.Add('Any user logs on');
          end;
          sI_Run : begin
            SubItems.Add('3');
            SubItems.Add('All users');
            SubItems.Add('Any user logs on');
          end;
          sRMacR : begin
            SubItems.Add('4');
            SubItems.Add('All users');
            SubItems.Add('Any user logs on');
          end;
          sRPerR : begin
            SubItems.Add('5');
            SubItems.Add('Current user only');
            SubItems.Add('Current user logs on');
          end;
          sCommn : begin
            SubItems.Add('6');
            SubItems.Add('All users');
            SubItems.Add('Any user logs on');
          end;
          sStart : begin
            SubItems.Add('7');
            SubItems.Add('Current user only');
            SubItems.Add('Current user logs on');
          end;
        end;
        SubItems.Add(cmd);
        SubItems.Add(vExt);
      end;
    IF LI.Index = Idx THEN
      begin
        lvProgs.Selected    := LI;
        lvProgs.ItemFocused := LI;
      end;
  end;

  procedure ILine(const S : String; src : TStSrcType; Light : Integer);
  // WIN.INI Load= and Run= lines contain one or more program
  // names in short filename form, separated by semicolons.
  // This function processes one such line
  VAR
    S1, S2 : String;
    P      : Integer;
  begin
    S1 := S;
    P  := Pos(';', S1);
    WHILE P > 0 DO
      begin
        S2 := Trim(Copy(S1, 1, P-1));
        AddAnItem(S2, ExtractFileName(S2), Src, Light, '');
        Delete(S1, 1, P);
        S1 := Trim(S1);
        P  := Pos(';', S1);
      end;
    IF S1 <> '' THEN
      AddAnItem(S1, ExtractFileName(S1), Src, Light, '');
  end;

  procedure RegSection(vRootKey : hKEY; const keyname : String;
    src : TStSrcType; Light : Integer);
  // Each value belonging to a Run-related Registry key has
  // a unique descriptive name. Its data is the command to run.
  VAR
    N   : Integer;
    rTS : TStringList;
  begin
    rTS := TStringList.Create;
    try
      WITH TRegistry.Create DO
      try
        RootKey := vRootKey;
        IF OpenKey(keyname, False) THEN
          begin
            GetValueNames(rTS);
            FOR N := 0 TO rTS.Count-1 DO
              AddAnItem(ReadString(rTS[N]), rTS[N], src, Light, '');
          end;
        CloseKey;
      finally
        Free;
      end;
    finally
      rTS.Free;
    end;
  end;

  procedure GetALink(const Path, S : String; src : TStSrcType;
    Light : Integer);
  VAR
    PBuff    : ARRAY[0..MAX_PATH] OF Char;
    Ext, Url : String;
  begin
    Ext := Uppercase(ExtractFileExt(S));
    IF Ext = '.URL' THEN
      begin
        Url := '';
        WITH TIniFile.Create(Path+S) DO
        try
          Url := ReadString('InternetShortcut', 'URL', '');
        finally
          Free;
        end;
        IF Url <> '' THEN
          AddAnItem(Url, Copy(S, 1, Length(S)-4), Src, Light, '.url')
        ELSE AddAnItem(Path+S, '*'+S, Src, Light, '');
      end
    ELSE IF Ext = '.LNK' THEN
      begin
        IF ReadALink(Path+S, PBuff, nil, nil, nil, nil, nil) THEN
          AddAnItem(StrPas(pBuff), Copy(S, 1, Length(S)-4), Src,
            Light, '.lnk')
        ELSE AddAnItem(Path+S, '*'+S, Src, Light, '');
      end
    ELSE AddAnItem(Path+S, '*'+S, Src, Light, '');
  end;

  procedure ReadShortcuts(nfolder : Integer; src : TStSrcType);
  VAR
    thePATH : String;
    TS      : TSearchRec;
    Rslt    : Integer;
  begin
    // Get enabled items
    thePath := GetSpecialPath(Handle, nFolder);
    IF thePath = '' THEN Exit;
    Rslt := FindFirst(FinalSlash(thePATH)+'*.*', faAnyFile, TS);
    WHILE Rslt = 0 DO
      begin
        IF (TS.Attr <> faDirectory) OR (TS.Name[1] <> '.') THEN
          GetALink(FinalSlash(thePATH), TS.Name, src, 1);
        Rslt := FindNext(TS);
      end;
    FindClose(TS);
    // Get disabled items
    thePATH := FinalSlash(ExtractFileDir(thePATH)) + DisStart;
    Rslt := FindFirst(FinalSlash(thePATH)+'*.*', faAnyFile, TS);
    WHILE Rslt = 0 DO
      begin
        IF (TS.Attr <> faDirectory) OR (TS.Name[1] <> '.') THEN
          GetALink(FinalSlash(thePATH), TS.Name, src, 2);
        Rslt := FindNext(TS);
      end;
    FindClose(TS);
    // Get items packed for removal
    thePATH := FinalSlash(ExtractFileDir(thePATH)) + RemStart;
    Rslt := FindFirst(FinalSlash(thePATH)+'*.*', faAnyFile, TS);
    WHILE Rslt = 0 DO
      begin
        IF (TS.Attr <> faDirectory) OR (TS.Name[1] <> '.') THEN
          GetALink(FinalSlash(thePATH), TS.Name, src, 3);
        Rslt := FindNext(TS);
      end;
    FindClose(TS);
  end;

  procedure CheckIfFirstTime;
  // If this is the first time Startup Cop has been launched
  // during the current Windows session, record all items
  // that are marked as packed for removal in a special list.
  // The user will be permitted to delete these items.
  VAR
    N   : Integer;
    key : String;
  begin
    IF GlobalFindAtom(AtomName) = 0 THEN
      begin
        GlobalAddAtom(AtomName);
        WITH TIniFile.Create(IniName) DO
        try
          EraseSection('Removals');
          FOR N := 0 TO lvProgs.Items.Count-1 DO
            WITH lvProgs.Items[N] DO
              IF StateIndex = 3 THEN
                begin
                  WriteString('Removals', Format('%d%s',[ImageIndex,
                    Caption]), SubItems[3]);
                  StateIndex := 4;
                  Data       := Pointer(4);
                end;
        finally
          Free;
        end;
      end
    ELSE
      begin
        WITH TIniFile.Create(IniName) DO
        try
          FOR N := 0 TO lvProgs.Items.Count-1 DO
            WITH lvProgs.Items[N] DO
              begin
                IF StateIndex < 3 THEN Continue;
                key := Format('%d%s', [ImageIndex, Caption]);
                IF ReadString('Removals', key, '') = SubItems[3] THEN
                  begin
                    StateIndex := 4;
                    Data       := Pointer(4);
                  end;
              end;
        finally
          Free;
        end;
      end;
  end;

  procedure CleanUpDupes;
  VAR N : Integer;
    procedure DelIni(I : TListItem);
    // Delete the INI item, because it's a dupe. If deletion leaves
    // the source of a non-enabled item empty, remove the source.
    VAR
      fmKey     : String;
      iVal, cmd : String;
      P, L      : Integer;
    begin
      WITH I DO
        begin
          cmd        := subItems[3];
          CASE TStSrcType(ImageIndex) OF
            sILoad : begin
              CASE StateIndex OF
                1   : fmKey := 'Load';
                2   : fmKey := 'NoLoad';
                3,4 : fmKey := 'RemLoad';
              END;
            end;
            sI_Run : begin
              CASE StateIndex OF
                1   : fmKey := 'Run';
                2   : fmKey := 'NoRun';
                3,4 : fmKey := 'RemRun';
              END;
            end;
            ELSE Exit; // impossible image index
          END;
          // Flush the cached copy of WIN.INI, just in case
          WritePrivateProfileString(nil, nil, nil, 'WIN.INI');
          WITH TIniFile.Create('WIN.INI') DO
          try
            iVal := ReadString('Windows', fmKey, '');
            L    := Length(cmd)+1;
            P    := Pos(cmd+';', iVal);
            IF P = 0 THEN P := Pos(';'+cmd, iVal);
            IF P = 0 THEN Dec(L);
            IF P = 0 THEN P := Pos(cmd, iVal);
            IF P <> 0 THEN
              begin
                System.Delete(iVal, P, L);
                iVal := Trim(iVal);
                // If the fmKey is non-enabled and is now empty,
                // delete it
                IF (Integer(Data) > 1) AND (iVal = '') THEN
                  DeleteKey('Windows', fmKey)
                ELSE WriteString('Windows', fmKey, iVal);
              end;
          finally
            Free;
          end;
        end;
    end;

    procedure DelReg(I : TListItem);
    // Delete the registry value from the key that now contains it.
    // If the item was disabled and the "minus" key is now empty,
    // delete it
    VAR
      rootk        : hKey;
      vName, vData : String;
      fmKey        : String;
      RKI          : TRegKeyInfo;
    begin
      Exit;
      WITH I DO
        begin
          vName := Caption;
          vData := SubItems[3];
          CASE TStSrcType(ImageIndex) OF
            sRMacS : begin
              rootk      := HKEY_LOCAL_MACHINE;
              CASE StateIndex OF
                1   : fmKey := KeySvc;
                2   : fmKey := NotSvc;
                3,4 : fmKey := RemSvc;
              END;
            end;
            sRMacR : begin
              rootk      := HKEY_LOCAL_MACHINE;
              CASE StateIndex OF
                1   : fmKey := KeyRun;
                2   : fmKey := NotRun;
                3,4 : fmKey := RemRun;
              END;
            end;
            sRPerR : begin
              rootk      := HKEY_CURRENT_USER;
              CASE StateIndex OF
                1   : fmKey := KeyRun;
                2   : fmKey := NotRun;
                3,4 : fmKey := RemRun;
              END;
            end;
            ELSE Exit; // impossible ImageIndex
          END;
          WITH TRegistry.Create DO
          try
            RootKey := rootk;
            IF OpenKey(fmKey, False) AND ValueExists(vName) THEN
              begin
                DeleteValue(vName);
                // If the source is non-enabled and empty, delete it
                IF Integer(Data) > 1 THEN
                  begin
                    GetKeyInfo(RKI);
                    IF RKI.NumValues = 0 THEN
                      begin
                        CloseKey;
                        IF OpenKey(MSWINCUR, False) THEN
                          CASE Integer(Data) OF
                            2 : CASE TStSrcType(ImageIndex) OF
                                  sRMacS : DeleteKey('RunServices-');
                                  sRMacR,
                                  sRPerR : DeleteKey('Run-');
                                END;
                            3,4 : CASE TStSrcType(ImageIndex) OF
                                  sRMacS : DeleteKey('RunServicesX');
                                  sRMacR,
                                  sRPerR : DeleteKey('RunX');
                                END;
                          END;
                      end;
                  end;
              end
          finally
            Free;
          end;
        end;
    end;

    procedure DelLnk(I : TListItem);
    VAR
      nFolder : Integer;
      thePIDL : PItemIDList;
      theBUFF : ARRAY[0..MAX_PATH] OF Char;
      fmPath  : String;
      fmName  : String;
      lnkName : String;
    begin
      WITH I DO
        begin
          IF (Caption <> '') AND (Caption[1] = '*') THEN
            lnkName := Copy(Caption, 2, 255)
          ELSE
            lnkName := Caption + SubItems[4];
          CASE TStSrcType(ImageIndex) OF
            sStart : nFolder := CSIDL_STARTUP;
            sCommn : nFolder := CSIDL_COMMON_STARTUP;
            ELSE Exit; // impossible imageIndex
          END;
          SHGetSpecialFolderLocation(Handle, nfolder, thePIDL);
          SHGetPathFromIDList(thePIDL, theBUFF);
          fmPath := StrPas(theBuff);
          CASE StateIndex OF
            1   : ;
            2   : fmPath := FinalSlash(ExtractFileDir(fmPATH)) +
                    DisStart;
            3,4 : fmPath := FinalSlash(ExtractFileDir(fmPATH)) +
                    RemStart;
          END;
          fmName := FinalSlash(fmPath)+lnkName;
          IF FileExists(fmName) OR DirectoryExists(fmName) THEN
            begin
              DeleteFile(fmName);
              RemoveDirectory(PChar(fmName));
              // If fmkey non-enabled, try to remove it. Will only
              // succeed if it's empty
              IF Integer(Data) > 1 THEN
                RemoveDirectory(PChar(fmPath));
            end;
        end;
    end;

  begin
    FOR N := lvProgs.Items.Count-1 DOWNTO 0 DO
      WITH lvProgs.Items[N] DO
        IF StateIndex > 10 THEN
          begin
            StateIndex := StateIndex - 10;
            CASE TStSrcType(ImageIndex) OF
              sILoad, sI_Run         : DelINI(LvProgs.Items[N]);
              sRMacS, sRMacR, sRPerR : DelREG(LvProgs.Items[N]);
              sStart, sCommn         : DelLNK(LvProgs.Items[N]);
            END;
            lvProgs.Items.Delete(N);
          end;
  end;

begin
  NumEna := 0;
  NumDis := 0;
  IF lvProgs.Selected = nil THEN
    Idx := -1
  ELSE Idx := lvProgs.Selected.Index;
  lvProgs.Items.Clear;
  Already := TStringList.Create;
  try
    // Get programs from HKLM\RunServices
    Already.Clear;
    RegSection(HKEY_LOCAL_MACHINE, KeySvc, sRMacS, 1); // enabled
    RegSection(HKEY_LOCAL_MACHINE, NotSvc, sRMacS, 2); // disabled
    RegSection(HKEY_LOCAL_MACHINE, RemSvc, sRMacS, 3); // packed
    // Flush the cached copy of WIN.INI, just in case
    WritePrivateProfileString(nil, nil, nil, 'WIN.INI');
    // Get programs from WIN.INI
    WITH TIniFile.Create('win.ini') DO
    try
      Already.Clear;
      ILine(ReadString('Windows','Load'  ,''),  sILoad, 1); // enabled
      ILine(ReadString('Windows','NoLoad',''),  sILoad, 2); // disabled
      ILine(ReadString('Windows','RemLoad',''), sILoad, 3); // packed
      Already.Clear;
      ILine(ReadString('Windows','Run'   ,''),  sI_Run, 1); // enabled
      ILine(ReadString('Windows','NoRun' ,''),  sI_Run, 2); // disabled
      ILine(ReadString('Windows','RemRun' ,''), sI_Run, 3); // packed
    finally
      Free;
    end;
    // Get programs from HKLM\Run
    Already.Clear;
    RegSection(HKEY_LOCAL_MACHINE, KeyRun, sRMacR, 1); // enabled
    RegSection(HKEY_LOCAL_MACHINE, NotRun, sRMacR, 2); // disabled
    RegSection(HKEY_LOCAL_MACHINE, RemRun, sRMacR, 3); // packed
    // Get programs from HKCU\Run
    Already.Clear;
    RegSection(HKEY_CURRENT_USER, KeyRun,  sRPerR, 1); // enabled
    RegSection(HKEY_CURRENT_USER, NotRun,  sRPerR, 2); // disabled
    RegSection(HKEY_CURRENT_USER, RemRun,  sRPerR, 3); // packed
    // Get programs from common startup
    Already.Clear;
    ReadShortcuts(CSIDL_COMMON_STARTUP, sCommn);
    // Get programs from personal startup
    Already.Clear;
    ReadShortcuts(CSIDL_STARTUP, sStart);
    // Clean up duplicate entries
    CleanUpDupes;
    // Size the columns
    CheckIfFirstTime;
    lvProgs.Perform(LVM_SETCOLUMNWIDTH, 0, LVSCW_AUTOSIZE);
    lvProgs.Perform(LVM_SETCOLUMNWIDTH, 1, LVSCW_AUTOSIZE_USEHEADER);
    lvProgs.Perform(LVM_SETCOLUMNWIDTH, 2, LVSCW_AUTOSIZE);
    lvProgs.Perform(LVM_SETCOLUMNWIDTH, 3, LVSCW_AUTOSIZE);
    try
      lvProgs.SetFocus;
    except
      ON Exception DO;
    end;
  finally
    Already.Free;
  end;
end;

procedure TMainForm.MakeChanges(TellTrouble : Boolean);
VAR N, numDels, numPacks : Integer;
CONST
  plu1 : ARRAY[Boolean] OF String[6] = ('s have', ' has');
  plu2 : ARRAY[Boolean] OF String[9] = ('ese items', 'is item');
  plu3 : ARRAY[Boolean] OF String[6] = ('them', 'it');
  plu4 : ARRAY[Boolean] OF String[1] = ('s', '');

  procedure DoINI(I : TListItem);
  // Move the INI key from its current location to its new location,
  // which may be limbo. If moving leaves the source of a non-enabled
  // item empty, remove the source.
  VAR
    fmKey, toKey : String;
    iVal, cmd    : String;
    P, L         : Integer;
  begin
    WITH I DO
      begin
        cmd        := subItems[3];
        CASE TStSrcType(ImageIndex) OF
          sILoad : begin
            CASE Integer(Data) OF
              1   : fmKey := 'Load';
              2   : fmKey := 'NoLoad';
              3,4 : fmKey := 'RemLoad';
            END;
            CASE StateIndex OF
              1   : toKey := 'Load';
              2   : toKey := 'NoLoad';
              3,4 : toKey := 'RemLoad';
              5   : toKey := '';
            END;
          end;
          sI_Run : begin
            CASE Integer(Data) OF
              1   : fmKey := 'Run';
              2   : fmKey := 'NoRun';
              3,4 : fmKey := 'RemRun';
            END;
            CASE StateIndex OF
              1   : toKey := 'Run';
              2   : toKey := 'NoRun';
              3,4 : toKey := 'RemRun';
              5   : tokey := '';
            END;
          end;
          ELSE Exit; // impossible image index
        END;
        // Flush the cached copy of WIN.INI, just in case
        WritePrivateProfileString(nil, nil, nil, 'WIN.INI');
        WITH TIniFile.Create('WIN.INI') DO
        try
          iVal := ReadString('Windows', fmKey, '');
          L    := Length(cmd)+1;
          P    := Pos(cmd+';', iVal);
          IF P = 0 THEN P := Pos(';'+cmd, iVal);
          IF P = 0 THEN Dec(L);
          IF P = 0 THEN P := Pos(cmd, iVal);
          IF P <> 0 THEN
            begin
              System.Delete(iVal, P, L);
              iVal := Trim(iVal);
              // If the fmKey is non-enabled and is now empty,
              // delete it
              IF (Integer(Data) > 1) AND (iVal = '') THEN
                DeleteKey('Windows', fmKey)
              ELSE WriteString('Windows', fmKey, iVal);
              IF toKey <> '' THEN
                begin
                  iVal := ReadString('Windows', toKey, '');
                  IF iVal <> '' THEN iVal := iVal + ';';
                  iVal := iVal + cmd;
                  WriteString('Windows', toKey, iVal);
                end;
            end
          ELSE IF TellTrouble THEN
            MessageBox(Handle, PChar(Format('The WIN.INI %s= entry '+
              'does not contain the command "%s". WIN.INI must '+
              'have been modified while Startup Cop was running.',
              [fmKey, cmd])), 'Startup Cop',
              MB_OK OR MB_ICONWARNING);
        finally
          Free;
        end;
      end;
  end;

  procedure DoREG(I : TListItem);
  // Delete the registry value from the key that now contains it
  // and, if Del is false, add it to the opposite key. If the
  // item was disabled and the "minus" key is now empty, delete it
  VAR
    rootk        : hKey;
    vName, vData : String;
    fmKey, toKey : String;
    RKI          : TRegKeyInfo;
  begin
    WITH I DO
      begin
        vName := Caption;
        vData := SubItems[3];
        CASE TStSrcType(ImageIndex) OF
          sRMacS : begin
            rootk      := HKEY_LOCAL_MACHINE;
            CASE Integer(Data) OF
              1   : fmKey := KeySvc;
              2   : fmKey := NotSvc;
              3,4 : fmKey := RemSvc;
            END;
            CASE StateIndex OF
              1   : toKey := KeySvc;
              2   : toKey := NotSvc;
              3,4 : toKey := RemSvc;
              5   : tokey := '';
            END;
          end;
          sRMacR : begin
            rootk      := HKEY_LOCAL_MACHINE;
            CASE Integer(Data) OF
              1   : fmKey := KeyRun;
              2   : fmKey := NotRun;
              3,4 : fmKey := RemRun;
            END;
            CASE StateIndex OF
              1   : toKey := KeyRun;
              2   : toKey := NotRun;
              3,4 : toKey := RemRun;
              5   : tokey := '';
            END;
          end;
          sRPerR : begin
            rootk      := HKEY_CURRENT_USER;
            CASE Integer(Data) OF
              1   : fmKey := KeyRun;
              2   : fmKey := NotRun;
              3,4 : fmKey := RemRun;
            END;
            CASE StateIndex OF
              1   : toKey := KeyRun;
              2   : toKey := NotRun;
              3,4 : toKey := RemRun;
              5   : tokey := '';
            END;
          end;
          ELSE Exit; // impossible ImageIndex
        END;
        WITH TRegistry.Create DO
        try
          RootKey := rootk;
          IF OpenKey(fmKey, False) AND ValueExists(vName) THEN
            begin
              DeleteValue(vName);
              // If the source is non-enabled and empty, delete it
              IF Integer(Data) > 1 THEN
                begin
                  GetKeyInfo(RKI);
                  IF RKI.NumValues = 0 THEN
                    begin
                      CloseKey;
                      IF OpenKey(MSWINCUR, False) THEN
                        CASE Integer(Data) OF
                          2 : CASE TStSrcType(ImageIndex) OF
                                sRMacS : DeleteKey('RunServices-');
                                sRMacR,
                                sRPerR : DeleteKey('Run-');
                              END;
                          3,4 : CASE TStSrcType(ImageIndex) OF
                                sRMacS : DeleteKey('RunServicesX');
                                sRMacR,
                                sRPerR : DeleteKey('RunX');
                              END;
                        END;
                    end;
                end;
              IF toKey <> '' THEN
                begin
                  CloseKey;
                  OpenKey(toKey, True);
                  WriteString(vName, vData);
                end;
            end
          ELSE IF TellTrouble THEN
            MessageBox(Handle, PChar(Format('The %s key does not'+
              'contain a value named %s. The Registry must '+
              'have been modified while Startup Cop was running.',
              [fmKey, vName])), 'Startup Cop',
              MB_OK OR MB_ICONWARNING);
        finally
          Free;
        end;
      end;
  end;

  procedure DoLNK(I : TListItem);
  VAR
    nFolder        : Integer;
    thePIDL        : PItemIDList;
    theBUFF        : ARRAY[0..MAX_PATH] OF Char;
    fmPath, toPath : String;
    fmName, toName : String;
    lnkName        : String;
  begin
    WITH I DO
      begin
        IF (Caption <> '') AND (Caption[1] = '*') THEN
          lnkName := Copy(Caption, 2, 255)
        ELSE
          lnkName := Caption + SubItems[4];
        CASE TStSrcType(ImageIndex) OF
          sStart : nFolder := CSIDL_STARTUP;
          sCommn : nFolder := CSIDL_COMMON_STARTUP;
          ELSE Exit; // impossible imageIndex
        END;
        SHGetSpecialFolderLocation(Handle, nfolder, thePIDL);
        SHGetPathFromIDList(thePIDL, theBUFF);
        fmPath := StrPas(theBuff);
        toPath := fmPath;
        CASE Integer(Data) OF
          1   : ;
          2   : fmPath := FinalSlash(ExtractFileDir(fmPATH)) +
                  DisStart;
          3,4 : fmPath := FinalSlash(ExtractFileDir(fmPATH)) +
                  RemStart;
        END;
        CASE StateIndex OF
          1   : ;
          2   : toPath := FinalSlash(ExtractFileDir(fmPATH)) +
                  DisStart;
          3,4 : toPath := FinalSlash(ExtractFileDir(fmPATH)) +
                  RemStart;
          5   : topath := '';
        END;
        fmName := FinalSlash(fmPath)+lnkName;
        IF toPath = '' THEN toName := ''
        ELSE toName := FinalSlash(toPath)+lnkName;
        IF FileExists(fmName) OR DirectoryExists(fmName) THEN
          begin
            IF toName = '' THEN
              begin
                DeleteFile(fmName);
                RemoveDirectory(PChar(fmName));
              end
            ELSE
              begin
                // Create destination (ok if already exists)
                CreateDirectory(PChar(toPath), nil);
                // If destination non-enabled, hide it
                IF StateIndex > 1 THEN
                  SetFileAttributes(PChar(toPath), GetFileAttributes(
                    PChar(toPath)) OR FILE_ATTRIBUTE_HIDDEN);
                IF NOT MoveFile(PChar(fmName), PChar(toName)) THEN
                  IF GetLastError = ERROR_ALREADY_EXISTS THEN
                    IF DeleteFile(PChar(toName)) THEN
                      IF NOT MoveFile(PChar(fmName), PChar(toName)) THEN
                        IF TellTrouble THEN
                          ShowMessage(SysErrorMessage(GetLastError));
              end;
            // If fmkey non-enabled, try to remove it. Will only
            // succeed if it's empty
            IF Integer(Data) > 1 THEN
              RemoveDirectory(PChar(fmPath));
          end
        ELSE IF TellTrouble THEN
          MessageBox(Handle, PChar(Format('The shortcut %s'+
            'does not exist. It must have been deleted or '+
            'renamed while Startup Cop was running.',
            [fmName])), 'Startup Cop',
            MB_OK OR MB_ICONWARNING);
      end;
  end;

  procedure RemoveRemovals;
  VAR
    N   : Integer;
    Key : String;
  begin
    WITH TIniFile.Create(IniName) DO
    try
      FOR N := 0 TO lvProgs.Items.Count-1 DO
        WITH lvProgs.Items[N] DO
          IF (Integer(Data)=4) AND (StateIndex <> 4) THEN
            begin
              key := Format('%d%s', [ImageIndex, Caption]);
              IF ReadString('Removals', key, '') = SubItems[3] THEN
                DeleteKey('Removals', key);
            end;
    finally
      Free;
    end;
  end;

begin
  NumDels := 0; NumPacks := 0;
  FOR N := 0 TO lvProgs.Items.Count-1 DO
    WITH lvProgs.items[N] DO
      IF (StateIndex = 3) AND (Integer(Data) < 3) THEN
        Inc(NumPacks)
      ELSE IF StateIndex = 5 THEN
        Inc(NumDels);
  IF NumDels > 0 THEN
    begin
      MessageBeep(MB_ICONWARNING);
      IF MessageBox(Handle, PChar(Format('You have marked %d startu'+
        'p item%s for permanent removal.'#13#10#13#10'This action i'+
        's not reversible. If you remove th%s, you will not be able'+
        ' to restore %s using Startup Cop. '#13#10#13#10'You should'+
        ' only remove entries corresponding to programs that you ha'+
        've uninstalled from your system.'#13#10#13#10'Do you want '+
        'to proceed?', [NumDels, plu4[NumDels=1], plu2[NumDels=1],
        plu3[NumDels=1]])), 'Startup Cop Warning', MB_YESNO OR
        MB_ICONWARNING) <> idYES THEN Exit;
    end;
  RemoveRemovals;
  FOR N := 0 TO lvProgs.Items.Count-1 DO
    begin
      WITH lvProgs.Items[N] DO
        IF StateIndex <> Integer(Data) THEN
          CASE TStSrcType(ImageIndex) OF
            sILoad, sI_Run         : DoINI(LvProgs.Items[N]);
            sRMacS, sRMacR, sRPerR : DoREG(LvProgs.Items[N]);
            sStart, sCommn         : DoLNK(LvProgs.Items[N]);
          END;
    end;
  IF NumPacks > 0 THEN
    begin
      MessageBeep(MB_ICONINFORMATION);
      MessageBox(Handle, PChar(Format('%d startup item%s been '+
        '"packed" for removal.'#13#10#13#10'After you restart Windo'+
        'ws and confirm that everything still works, you will find '+
        'th%s marked with a red-light and question-mark icon.'#13#10+
        #13#10'When you select such an item, you will be permitted '+
        'to remove the item permanently.', [Numpacks,
        plu1[Numpacks=1], plu2[Numpacks=1]])), 'Startup Cop',
        MB_OK OR MB_ICONINFORMATION);
    end;
end;

function TMainForm.IsSysTray(TLI : TListItem) : Boolean;
begin
  Result := False;
  IF TLI.ImageIndex <> Integer(srMacR) THEN Exit;
  IF AnsiCompareText(ExtractFileName(TLI.SubItems[3]),
    'systray.exe') <> 0 THEN Exit;
  Result := True;
end;

function TMainForm.RestoreProfile(const Sec : String) : Integer;
VAR
  SL       : TStringList;
  EnDis,
  NotEnDis : Integer;
  N        : Integer;
  key      : String;
  SavedCmd : String;
begin
  SL := TStringList.Create;
  try
    WITH TIniFile.Create(Ininame) DO
    try
      // If the profile doesn't exist, return -1
      ReadSections(SL);
      IF SL.IndexOf(Sec) < 0 THEN
        begin
          Result := -1;
          Exit;
        end;
      // If doesn't have a State= key, return -2
      EnDis := ReadInteger(Sec, 'State', -1);
      IF EnDis = -1 THEN
        begin
          Result := -2;
          Exit;
        end;
      IF EnDis <> 1 THEN EnDis := 2;
      NotEnDis := 3-EnDis;
      // If the profile is empty, return -3
      ReadSection(sec, SL);
      IF SL.Count <= 0 THEN
        begin
          Result := -3;
          Exit;
        end;
      Result := 0;
      FOR N := 0 TO lvProgs.Items.Count-1 DO
        WITH lvProgs.Items[N] DO
          begin
            IF IsSysTray(lvProgs.Items[N]) THEN
              begin
                StateIndex := 1;
                Continue;
              end;
            key := Format('%d%s', [ImageIndex, Caption]);
            // Don't change items that are packed for removal
            IF StateIndex >= 3 THEN Continue;
            IF SL.IndexOf(key) >= 0 THEN
              begin
                SavedCmd := ReadString(sec, key, '');
                IF SavedCmd = SubItems[3] THEN
                  StateIndex := EnDis
                // New code begin (1.0.1)
                // The GetPrivateProfileString() function (called by
                // ReadString() above) strips delimiting double-quotes.
                ELSE IF Format('"%s"', [SavedCmd]) = SubItems[3] THEN
                  StateIndex := EnDis
                // New code end
                ELSE StateIndex := NotEnDis;
              end
            ELSE StateIndex := NotEnDis;
          end;
    finally
      Free;
    end;
  finally
    SL.Free;
  end;
end;

procedure TMainForm.DefaultHandler(VAR Message);
begin
  Inherited DefaultHandler(Message);
  WITH TMessage(Message) DO
    IF Msg = CmdLineMsg THEN
      begin
        IF wParam = 0 THEN ProcessCmdLine
        ELSE Close;
      end;
end;

end.






